/**
 * 
 */
package gov.va.med.mhv.usermgmt.integration.service.util;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.TimestampUtils;
import gov.va.med.mhv.integration.phr.transfer.FacilityExtractStatus;
import gov.va.med.mhv.integration.phr.transfer.PatientExtractStatus;
import gov.va.med.mhv.integration.phr.transfer.PatientIdentifier;
import gov.va.med.mhv.integration.phr.transfer.RefreshDataParameters;
import gov.va.med.mhv.usermgmt.enumeration.ExtractType;
import gov.va.med.mhv.usermgmt.transfer.Facility;
import gov.va.med.mhv.usermgmt.transfer.Patient;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public final class PhrInterfaceUtils {
    
    private static final Log LOG = LogFactory.getLog(PhrInterfaceUtils.class);

    /**
     * 
     */
    private PhrInterfaceUtils() {
    }
    
    public static String describe(RefreshDataParameters parameters) {
        StringBuilder builder = new StringBuilder();
        if (parameters != null) {
            builder.append("RefreshDataParameters@");
            builder.append(Integer.toHexString(parameters.hashCode()));
            builder.append("[Patient=");
            builder.append(describe(parameters.getPatient()));
            builder.append("]");
        } else {
            builder.append("<No RefreshDataParameters>");
        }
        return builder.toString();
    }

    public static String describe(PatientIdentifier patient) {
        StringBuilder builder = new StringBuilder();
        if (patient != null) {
            builder.append("PatientIdentifier@");
            builder.append(Integer.toHexString(patient.hashCode()));
            builder.append("[ICN=");
            builder.append(patient.getIcn());
            builder.append("]");
        } else {
            builder.append("<No Patientidentifier>");
        }
        return builder.toString();
    }

    public static String describe(PatientExtractStatus patientExtractStatus) {
        StringBuilder builder = new StringBuilder();
        if (patientExtractStatus != null) {
            builder.append("PatientExtractStatus@");
            builder.append(Integer.toHexString(patientExtractStatus.hashCode()));
            builder.append("[Patient=");
            builder.append(describe(patientExtractStatus.getPatient()));
            builder.append(", Retrieved Date=");
            builder.append(patientExtractStatus.getRetrievedDate());
            builder.append(", Facility Statusses=");
            append(builder, patientExtractStatus.getFacilityExtractStatusList());
            builder.append("]");
        } else {
            builder.append("<No PatientExtractStatus>");
        }
        return builder.toString();
    }

    private static void append(StringBuilder builder,
        Collection<FacilityExtractStatus> extractStatusses) 
    {
        if (extractStatusses != null) {
            builder.append("{");
            boolean isFirst = true;
            for (FacilityExtractStatus status: extractStatusses) {
                if (isFirst) {
                    isFirst = false;
                } else {
                    builder.append(", ");
                }
                append(builder, status);
            }
            builder.append("}");
        } else {
            builder.append("null");
        }
    }
    
    public static String describe(FacilityExtractStatus extractStatus) {
        StringBuilder builder = new StringBuilder();
        append(builder, extractStatus);
        return builder.toString();
    }
    
    private static void append(StringBuilder builder, 
        FacilityExtractStatus extractStatus) 
    {
        if (extractStatus != null) {
            builder.append("FacilityExtractStatus@");
            builder.append(Integer.toHexString(extractStatus.hashCode()));
            builder.append("[Extract=");
            builder.append(extractStatus.getExtract());
            builder.append(", StationNumber=");
            builder.append(extractStatus.getStationNumber());
            builder.append(", ICN=");
            builder.append(extractStatus.getIcn());
            builder.append(", LastRequested=");
            builder.append(extractStatus.getLastRequested());
            builder.append(", LastModified=");
            builder.append(extractStatus.getLastModified());
            builder.append(", LastCompleted=");
            builder.append(extractStatus.getLastCompleted());
            builder.append("]");
        } else {
            builder.append("<No FacilityExtractStatus>");
        }
    }

    public static Collection<gov.va.med.mhv.usermgmt.transfer.
        FacilityExtractStatus> toFacilityExtractStatusses(Patient patient, 
        ExtractType extractType, PatientExtractStatus status)
    {
        Collection<gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus>
            statusses = null;
        if ((status != null)  
            && (status.getFacilityExtractStatusList() != null))
        {
            statusses = new ArrayList<gov.va.med.mhv.usermgmt.transfer.
                FacilityExtractStatus>();
            for (FacilityExtractStatus facilityStatus: 
                status.getFacilityExtractStatusList())
            {
                if ((facilityStatus != null) && 
                    ((extractType == null) || extractType.getName().equals(
                        facilityStatus.getExtract())))
                {
                    gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus 
                        facilityExtractStatus = PhrInterfaceUtils.
                        toFacilityExtractStatus(patient, status, facilityStatus);
                    statusses.add(facilityExtractStatus);
                } else {
                    // Not a status or not a status of interest
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Ignoring " + getUserName(patient) +
                            "'s facilityStatus " + describe(facilityStatus) 
                            + ", because not " + ((extractType != null) 
                               ? "for '" + extractType.getName() + "' extract"
                               : "a status"));
                    }
                }
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Created extract statusses " + describe(statusses) + 
                ", for " + getUserName(patient) +  
                ((extractType != null) 
                   ? " and '" + extractType.getName() + "' extract" : ""));
        }
        return statusses;
    }
    
    public static Collection<gov.va.med.mhv.usermgmt.transfer.
    FacilityExtractStatus> toFacilityExtractStatusses(Patient patient, 
    PatientExtractStatus status)
	{
	    Collection<gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus>
	        statusses = null;
	    if ((status != null)  
	        && (status.getFacilityExtractStatusList() != null))
	    {
	        statusses = new ArrayList<gov.va.med.mhv.usermgmt.transfer.
	            FacilityExtractStatus>();
	        for (FacilityExtractStatus facilityStatus: 
	            status.getFacilityExtractStatusList())
	        {
	            if (facilityStatus != null){
	                gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus 
	                    facilityExtractStatus = PhrInterfaceUtils.
	                    toFacilityExtractStatus(patient, status, facilityStatus);
	                statusses.add(facilityExtractStatus);
	            } else {
	                // Not a status or not a status of interest
	                if (LOG.isDebugEnabled()) {
	                    LOG.debug("Ignoring " + getUserName(patient) +
	                        "'s facilityStatus " + describe(facilityStatus));
	                }
	            }
	        }
	    }
	    return statusses;
	}    
    
    private static gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus 
        toFacilityExtractStatus(Patient patient,
        PatientExtractStatus patientExtractStatus,
        FacilityExtractStatus integrationStatus) 
    {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("integrationStatus", integrationStatus);
        gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus
            status = new gov.va.med.mhv.usermgmt.transfer. 
            FacilityExtractStatus();
        status.setCreatedOn(TimestampUtils.createCurrentTime());
        status.setExtractType(ExtractType.getEnum(integrationStatus.
            getExtract()));
        status.setFacility(getFacility(patient, integrationStatus.
            getStationNumber()));

        // add comparison to check if the individual refresh request has been processed today
        // this is necessary to ensure that the status is inProgress while waiting for the refresh to start
        // condition for completion is that the request is made today and that the completion status is true

        status.setInProgress(null == integrationStatus.getLastRequested() || !(patientExtractStatus.isSameDate(new java.util.Date(),integrationStatus.getLastRequested()) && integrationStatus.isCompleted()));
        
        
        if(integrationStatus.getLastModified()!= null ) {
        	status.setLastUpdated(TimestampUtils.create(integrationStatus.getLastModified()));
        }	
        else {
        	status.setLastUpdated(TimestampUtils.create(integrationStatus.getLastCompleted()));	
        }
        
        // NOTE: We do not set the ID or PK, since the entity is intended
        // to be Transient
        return status;
    }

    private static Facility getFacility(Patient patient, String stationNumber) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotBlank("stationNumber", stationNumber);
        if ((patient.getFacilitys() != null)) {   
           for (Object f: patient.getFacilitys()) {
               Facility facility = (Facility) f;
               if (stationNumber.equals(facility.getName())) {
                   return facility;
               }
               facility = null;
           }
        }
        return null;
    }
    
    private static String describe(Collection<gov.va.med.mhv.usermgmt.transfer.
        FacilityExtractStatus> extractStatusses) 
    {
        StringBuilder builder = new StringBuilder();
        if (extractStatusses != null) {
            builder.append("{");
            boolean isFirst = true;
            for (gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus 
                extractStatus: extractStatusses) 
            {
                if (isFirst) {
                    isFirst = false;
                } else {
                    builder.append(", ");
                }
                append(builder, extractStatus);
            }
            builder.append("}");
        } else {
            builder.append("<null>");
        }
        return builder.toString();
    }
    
    private static void append(StringBuilder builder,
        gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus extractStatus) 
    {
        if (extractStatus != null) {
            builder.append("FacilityExtractStatus@");
            builder.append(Integer.toHexString(extractStatus.hashCode()));
            builder.append("[Extract=");
            builder.append(extractStatus.getExtractType());
            builder.append(", StationNumber=");
            builder.append((extractStatus.getFacility() != null) 
                ? extractStatus.getFacility().getName() : "-");
            builder.append(", CreatedOn=");
            builder.append(extractStatus.getCreatedOn());
            builder.append(", LastUpdated=");
            builder.append(extractStatus.getLastUpdated());
            builder.append("]");
        } else {
            builder.append("<No FacilityExtractStatus>");
        }
    }

    private static String getUserName(Patient patient) {
        return ((patient != null) && (patient.getUserProfile() != null))
            ? "patient '" + patient.getUserProfile().getUserName() + "'"
            : "<unknown patient>";
    }

}
